function rimless_wheel_mwa
% Minimum worked example of the rimless wheel model for legged locomotion.

%% Tabula Rasa
clear all           % clear all variables from workspace
close all           % close all open figures

%% Parameters
% Set up all the parameters needed for the simulation.

% The period of a pendulum is independent of the mass.

g = 9.8;            % gravity [m/s]
l = 1;              % leg length [m]
alpha = pi/8;		% half the angle between spokes [rad]
pitch = 0.15;		% pitch of the ramp [rad]

number_steps = 10;	% number of steps of the rimless wheel to simulate.

%% Initial Conditions

theta_dot = 4;      % start velocity [rad/s]

% At the start of each step both feet are in contact with the ramp. We
% therefore need to decide which foot will be the pivot foot by deciding
% whether the wheel is rolling up the slope or down the slope.
if theta_dot >= 0
	theta = pitch - alpha;
else
	theta = pitch + alpha;
end

%% Switching event

% We will use matlab's built in numerical solver to calculate the
% trajectory of the continuous phase. The continuous phase stops when the
% next foot touches the ramp, we can set the 'Events' option so that the
% solver can calculate the impact point too a high precision. @foot_impact
% indicates a function nested at the bottom of the page.

leg_switch = odeset('Events',@foot_impact);

%% Open figure for phase plot
figure
title('Phase Portrait Trajectories of a Rimless Wheel')
xlabel('\theta [rad]')
ylabel('d\theta/dt [rad/s]')
hold on         % this is needed to plot several steps on the same axes.

%% Phase plot simulation
% We set up a for loop to calculate the trajectory of one step of the
% rimless wheel.

for i = 1:number_steps
    
    % State vector: q = [theta  theta_dot]
    %
    % We record the trajectory of each step inside a structure 'Step'. The
    % structure contains a time vector 't' and a state vector 'q' which
    % contains the trajectory for each state.
    % 
	[Step(i).t,Step(i).q] = ode45(@EoM, [0 2], [theta theta_dot],leg_switch);
	%        ^         ^            ^            ^        ^          ^
    %      time    trajectory    Equation     initial  initial  termination
    %     vector     vector         of         angle   velocity  condition
    %                             Motion
    
    % Discrete dynamics: set the initial conditions for the next step.
    theta_dot = Step(i).q(end,2)*cos(2*alpha);
	
    if theta_dot >= 0
		theta = pitch - alpha;
	else
		theta = pitch + alpha;
    end	
	
    % Plot the trajectory of this step.
    plot(Step(i).q(:,1),Step(i).q(:,2));

end

%% Equation of Motion
function output = EoM(~,q)
	% Matlab's numerical solver can solve a system of first order
	% equations. We therefore need to decompose the second order equation
	% of motion into two first order equations i.e. state space form.
    
	output(1,1) = q(2);
	output(2,1) = g*sin(q(1))/l;
end

%% Switch condition

function [impact,isterminal,direction] = foot_impact(~,q)
    % This function detects a zero crossing, therefore the impact variable
    % needs to be zero when the next foot touches the ramp. The 'abs()' is
    % needed as we don't know if the wheel is rolling up the slope or down
    % the slope.
    impact = abs(q(1) - pitch) - alpha;
    
    % The terminate switch is used to stop the simulation. If set to zero
    % the crossing point is recorded but the integration continues.
	isterminal = 1;
    
    % This variable indicates the direction of the zero crossing that will
    % be detected:
    %     1 => negative -> positive
    %    -1 => positive -> negative
    %     0 => both
    direction = 1;
end

end